
/* Copyright (C) 2001-2009 Monotype Imaging Inc. All rights reserved. */

/* Confidential information of Monotype Imaging Inc. */

/* fs_stik.c */

#include "fs_itype.h"
#ifdef FS_RENDER
#ifdef FS_STIK

#define SIN_2_DEGREES 2287 /* in 16.16 fixed of course */
#define SIN_30_DEGREES 32768

static FS_VOID bisect(_DS_ FS_FIXED fx1, FS_FIXED fy1, FS_FIXED fx2, FS_FIXED fy2, FS_FIXED fx3, FS_FIXED fy3);
static FS_VOID draw_quad_sub(_DS_ FS_FIXED fx1, FS_FIXED fy1, FS_FIXED fx2, FS_FIXED fy2, FS_FIXED fx3, FS_FIXED fy3);
/****************************************************************/
/* find the intersection point <c> of given a point <x0,y0> with a
* unit tangent <p0>, with another point <x1,y1> with unit tangent <p1>
*
* assuming they intersect, for some <s> and <t> we have:
* x0 + s*p0.x == x1 + t*p1.x
* y0 + s*p0.y == y1 + t*p1.y
*
* re-arrange
* s*p0.x - t*p1.x == x1 - x0
* s*p0.y - t*p1.y == y1 - y0
*
* multiply
* s*p0.x*p1.y - t*p1.x*p1.y == p1.y*(x1 - x0)
* s*p0.y*p1.x - t*p1.y*p1.x == p1.x*(y1 - y0)
*
* subtract
* s*(p0.x * p1.y - p0.y * p1.x) == p1.y(x1 - x0) - p1.x(y1 - y0)
*
* divide
* s = (p1.y(x1 - x0) - p1.x(y1 - y0)) / (p0.x * p1.y - p0.y * p1.x)
*
* intersection point is therefore
* c->x = x0 + s*p0.x;
* c->y = y0 + s*p0.y;
*
* BUT OF COURSE THIS IS ALL HAPPENING IN FS_FIXED */

static int intersect(FS_FIXED x0, FS_FIXED y0, FIXED_VECTOR *p0, FS_FIXED x1, FS_FIXED y1, FIXED_VECTOR *p1, FIXED_VECTOR *c)
{
    FS_FIXED a, b, s;

    /* <p0> and <p1> are unit tangents ... <b> is the cross product which is the
    * sin of the included angle which is in the intvertal [-FIXED_ONE ... FIXED_ONE] */
    b = FixMul(p0->x, p1->y) - FixMul(p0->y, p1->x);

    /* unit tangents are almost parallel ... intersection will be bizarre
    * just use the midpoint of the chord from (x0,y0) to (x1,y1) */

    if (ABS(b) <= SIN_2_DEGREES)
    {
        c->x = (x0 + x1) / 2;
        c->y = (y0 + y1) / 2;
        return 1;
    }

    /* now we know we can subtract any two coordinates safely, and we're
     * multiplying by a value in the interval [-FIXED_ONE ... FIXED_ONE] so
     * there is no chance of overflow on <a> */
    a = FixMul(p1->y, x1 - x0) - FixMul(p1->x, y1 - y0);

    /* No assurances about overflow here ... BUT in the usage this was designed
     * for (finding the control point for the left/right offset parabola's of a
     * stroke) we should be OK */

    s = FixDiv(a, b);
    c->x = x0 + FixMul(s, p0->x);
    c->y = y0 + FixMul(s, p0->y);

    /* return OK if (x,y) aren't at infinity */
    return (ABS(c->x) != MYINFINITY && ABS(c->y) != MYINFINITY) ? 0 : 1;
}


/****************************************************************/
/* return index of v in v[] ... return <n> if not found */
static int indexof(FS_FIXED x, int n, FS_FIXED *v)
{
    int i;

    for (i = 0; i < n && x != v[i]; i++) ;
    return i;
}

/****************************************************************/
/* val lies between two values of oldv[], return value having the
* same percentage relationship with corresponding members of newv[]
*
* that is solve the following equation for <result>
*
*  result - old_lo   val - new_lo
*  --------------- = ---------------
*  old_hi - old_lo   new_hi - new_lo
*/
static FS_FIXED interp(FS_FIXED val, int num, FS_FIXED *oldv, FS_FIXED *newv)
{
    int i;
    FS_FIXED a, b, c, d;

    /* find largest <i> suchthat old[i] <= val */
    for (i = num - 1; num > 0 && oldv[i] > val; i--) ;

    if (val == oldv[i])
        return newv[i];

    a = newv[i + 1] - newv[i];
    b = val - oldv[i];
    c = oldv[i + 1] - oldv[i];
    d = newv[i] + LongMulDiv(a, b, c);
    return d;
}

/****************************************************************/
/* shell sort oldv[] and newv[] in ascending order by oldv[] */
static FS_VOID sort_hints(int n, FS_FIXED *oldv, FS_FIXED *newv)
{
    int gap, i, j;
    FS_FIXED temp;

    for (gap = n / 2; gap > 0; gap /= 2)
    {
        for (i = gap; i < n; i++)
        {
            for (j = i - gap; j >= 0 && oldv[j] > oldv[j + gap]; j -= gap)
            {
                temp = oldv[j];
                oldv[j] = oldv[j + gap];
                oldv[j + gap] = temp;
                temp = newv[j];
                newv[j] = newv[j + gap];
                newv[j + gap] = temp;
            }
        }
    }
#ifdef GRID_DEBUG
    dump_hints(n, oldv, newv);
#endif
}

/****************************************************************/
/* possibly add <oldv,newv> to <old_v[],new_v[] */
static FS_SHORT add_pair(FS_FIXED oldv, FS_FIXED newv, FS_SHORT num, FS_FIXED *old_v, FS_FIXED *new_v)
{
    /* ? not already there */
    if (num == indexof(oldv, num, old_v))
    {
        old_v[num] = oldv;
        new_v[num] = newv;
        num++;
    }
    return num;
}
/****************************************************************/
/* round coord <z> to grid or half grid */
static FS_FIXED odd_round(FS_FIXED z, int odd)
{
    if (odd)
    {
        /* half grid (1.5, 2.5, 3.5, 4.5, etc..) */
        z = 0x00008000 + (z & 0xFFFF0000);
    }
    else
    {
        /* grid (1, 2, 3, 4, etc..) */
        z = (z + 0x00008000) & 0xFFFF0000;
    }
    return z;
}

/****************************************************************/
/* adjust coord<z> so that the closer side of the expanded stroke is on the grid line  */
/* adjust z to cover the closer grid line */
static FS_FIXED atcg(FS_FIXED z, FS_FIXED sw)
{
    FS_FIXED right, left, dl, dr, dis;
    right = z + sw / 2;
    left = z - sw / 2;
    dl = FIXED_ROUND(left) - left;
    dr = FIXED_ROUND(right) - right;
    if (ABS(dl) < ABS(dr))
        dis = dl;
    else
        dis = dr;

    return z + dis;
}
/****************************************************************/
static FS_VOID set_bbox(int n_pts, FS_OUTLINE *outl)
{
    FS_FIXED lo_x, hi_x, lo_y, hi_y, x, y;
    FS_FIXED *px, *py;
    int i;

    if (n_pts == 0 || outl == 0)
        return;

    px = (FS_FIXED *)(outl->x);
    lo_x = hi_x = px[0];
    py = (FS_FIXED *)(outl->y);
    lo_y = hi_y = py[0];

    for (i = 1; i < n_pts; i++)
    {
        x = px[i];
        if (x > hi_x) hi_x = x;
        if (x < lo_x) lo_x = x;
        y = py[i];
        if (y > hi_y) hi_y = y;
        if (y < lo_y) lo_y = y;
    }
    outl->lo_x = lo_x;
    outl->hi_x = hi_x;
    outl->lo_y = lo_y;
    outl->hi_y = hi_y;
}

/****************************************************************/

/*
* Do some autohinting of these stick characters ... basically
* move points with a vertical or horizontal tangent, as well as
* xMax, yMax, xMin and yMin to a 'nice' point on the grid.
*
* if the stroke width is <odd> use a half-integer point, else
* an integer grid point.
*
* then interpolate all other points with respect to these guys.
*
*/

/****************************************************************/
FS_SHORT autohint_stik(_DS_ FS_OUTLINE *stik, FS_FIXED sw)
{
    int i, j;
    FS_SHORT num = 0, n_pts;
    FS_FIXED *old_v, *new_v;
    FS_BOOLEAN int_sw = 1;
    int sw_odd;
    FS_FIXED stroke = 0;
    FS_SHORT   sw_i = 0;
    FS_FIXED *x = (FS_FIXED *)(stik->x), *y = (FS_FIXED *)(stik->y), z;
    FS_FIXED fsw = MAX(FIXED_ONE, STATE.stroke_pct * STATE.lpm);


    if ( STATE.flags & FLAGS_GRAYSCALE)
        int_sw = 0; /* non-integer stroke width */
    else
    {
        stroke = (fsw + FIXED_ONEHALF) & 0xFFFF0000;
        sw_i = FS_FLOOR(stroke);
    }
    sw_odd = sw_i & 1;
    n_pts = stik->np;

    /* they don't want hints */
    if (STATE.flags & FLAGS_HINTS_OFF)
        return n_pts;

    /* not a real stik ... quit */
    if (n_pts < 2)
        return n_pts;

    /* if the stik had hints ... don't do any autohinting */
    if (STATE.any_hints & OUTL_FLAGS_ANYHINTS)
        return n_pts;


    /* need some temporary storage */
#ifdef FS_MEM_DBG
    STATE.memdbgid = "autohint(old_v)";
#endif
    old_v = (FS_FIXED *)FSS_malloc(_PS_ (2 + n_pts) * 4);
#ifdef FS_MEM_DBG
    STATE.memdbgid = "autohint(new_v)";
#endif
    new_v = (FS_FIXED *)FSS_malloc(_PS_ (2 + n_pts) * 4);
    if (old_v == 0 || new_v == 0)
    {
        FSS_free(_PS_ old_v);
        FSS_free(_PS_ new_v);
        return n_pts;
    }
    /* snap points to reference lines */
    if (STATE.use_reflines)
    {
        SFNT *sfnt = STATE.cur_typeset.tfntarray[STATE.cur_font].sfnt;
        FS_FIXED br, bs, xr, xs, cr, cs;
        FS_FIXED gbr, gbs, gxr, gxs, gcr, gcs;
        FS_FIXED fuzz = 1024;  /* 1/64-th pixel */
        /* y_heights extracted from font characters 'HOxo' */
        cr = sfnt->senv->cap_round;
        cs = sfnt->senv->cap_square;
        xr = sfnt->senv->x_round;
        xs = sfnt->senv->x_square;
        bs = sfnt->senv->base_square;
        br = sfnt->senv->base_round;
        /* round them to grid: specifically, round squares to  */
        /* grid, then move corresponding rounds an integer pixel */
        /* away. note we truncate, rather than round to keep the */
        /* rounds equal to the squares until the difference is */
        /* greater than or equal to 1 pixel */
        if (int_sw)
        {
            gcs = odd_round(cs, sw_odd);
            gxs = odd_round(xs, sw_odd);
            gbs = odd_round(bs, sw_odd);
            gcr = gcs + (0xFFFF0000 & (cr - cs));
            gxr = gxs + (0xFFFF0000 & (xr - xs));
            gbr = gbs - (0xFFFF0000 & (bs - br));
        }
        else
        {
            gcs = atcg(cs, sw);
            gxs = atcg(xs, sw);
            gbs = atcg(bs, sw);
            gcr = gcs + (0xFFFF0000 & (cr - cs));
            gxr = gxs + (0xFFFF0000 & (xr - xs));
            gbr = gbs - (0xFFFF0000 & (bs - br));
        }
        /* reduce x-height -- gotta be room for ij dot and space */
        if (gcs - gxs <= FIXED_ONE)
            gxs -= FIXED_ONE;
        if (gcr - gxr <= FIXED_ONE)
            gxr -= FIXED_ONE;
        /****************************************************
        *
        * NOTE: the above are rather different from these
        * even though they SHOULD be the same.  Some strange
        * "optimized" scaling code is responsible, look at
        * the use of macros FROUND() SROUND() versus FixMul()
        * FS_FIXED scl = STATE.cur_sfnt->user_scale[3];
        * scl /= STATE.cur_sfnt->lfnt->ttf->unitsPerEm;
        * cr = 169 * scl;
        * cs = 166 * scl;
        * xr = 119 * scl;
        * xs = 116 * scl;
        * bs = 15 * scl;
        * br = 12 * scl;
        *
        * eg: at 17 lpm, the differences are (FROUND vs scl)
        * 11.156250    11.22265625
        * 10.953123 11.02343750
        * 7.9687500  7.90234375
        * 7.7656250  7.70312500
        * 1.0000000  0.99609375
        * 0.7968750  0.79687500
        ******************************************************/
        y = (FS_FIXED *)(stik->y);
        for (num = 0, i = 0; i < n_pts; i++)
        {
            /* in bottom zone */
            if (br - fuzz <= y[i] && y[i] <= bs + fuzz)
            {
                if (ABS(y[i] - bs) < FIXED_ONEHALF)
                    num = add_pair(y[i], gbs, num, old_v, new_v);
                else if (ABS(y[i] - br) < FIXED_ONEHALF)
                    num = add_pair(y[i], gbr, num, old_v, new_v);
            }
            /* in cap zone */
            else if (cs - fuzz <= y[i] && y[i] <= cr + fuzz)
            {
                if (ABS(y[i] - cs) < FIXED_ONEHALF)
                    num = add_pair(y[i], gcs, num, old_v, new_v);
                else if (ABS(y[i] - cr) < FIXED_ONEHALF)
                    num = add_pair(y[i], gcr, num, old_v, new_v);
            }
            /* in x xone */
            else if (xs - fuzz <= y[i] && y[i] <= xr + fuzz)
            {
                if (ABS(y[i] - xs) < FIXED_ONEHALF)
                    num = add_pair(y[i], gxs, num, old_v, new_v);
                else if (ABS(y[i] - xr) < FIXED_ONEHALF)
                    num = add_pair(y[i], gxr, num, old_v, new_v);
            }
        }

        /* (maybe) add ymax, ymin */
        set_bbox(n_pts, stik);
        if (int_sw)
        {
            num = add_pair(stik->lo_y, odd_round(stik->lo_y, sw_odd), num, old_v, new_v);
            num = add_pair(stik->hi_y, odd_round(stik->hi_y, sw_odd), num, old_v, new_v);
        }
        else
        {
            num = add_pair(stik->lo_y, atcg(stik->lo_y, sw), num, old_v, new_v);
            num = add_pair(stik->hi_y, atcg(stik->hi_y, sw), num, old_v, new_v);
        }
        /* now interpolate other points wrt these */
        sort_hints(num, old_v, new_v);
        for (i = 0; i < n_pts; i++)
        {
            y[i] = interp(y[i], num, old_v, new_v);
        }
        set_bbox(n_pts, stik);
    }
    /* special case for dot's ... two point contours where the */
    /* y coordinates are within 1 pixel ... make then the same */
    for (i = 0, j = 0; i < stik->num; i++)
    {
        if (stik->type[i] == FS_MOVETO && j + 1 < n_pts && ABS(y[j] - y[j + 1]) < FIXED_ONE && x[j] == x[j + 1] )
        {
            /* really a 2 point contour */
            if ((i + 1 < stik->num && stik->type[i + 1] == FS_LINETO) &&
                    (i + 2 >= stik->num || stik->type[i + 2] == FS_MOVETO))
                y[j + 1] = y[j];
        }
        j += (stik->type[i] == FS_QUADTO) ? 2 : 1;
    }
    /*** this should be done 1 contour at a time ??? ***/

    /* now move interesting points to the grid, interpolate the rest */

    /** first for X **/
    for (num = 0, i = 0; i < n_pts; i++)
    {
        /* select the interesting points */
        if (i + 1 < n_pts && x[i] == x[i + 1])
        {
            /* add a new interesting point */
            if (int_sw)
                z = odd_round(x[i], sw_odd);
            else
                z = atcg(x[i], sw);
            num = add_pair(x[i], z, num, old_v, new_v);
        }
    }

    /* (maybe) add Xmax, Xmin */
    set_bbox(n_pts, stik);
    if (int_sw)
    {
        num = add_pair(stik->lo_x, odd_round(stik->lo_x, sw_odd), num, old_v, new_v);
        num = add_pair(stik->hi_x, odd_round(stik->hi_x, sw_odd), num, old_v, new_v);
    }
    else
    {
        num = add_pair(stik->lo_x, atcg(stik->lo_x, sw), num, old_v, new_v);
        num = add_pair(stik->hi_x, atcg(stik->hi_x, sw), num, old_v, new_v);
    }

    /* now interpolate other points wrt these */
    sort_hints(num, old_v, new_v);
    for (i = 0; i < n_pts; i++)
    {
        x[i] = interp(x[i], num, old_v, new_v);
    }

    /** now for Y **/
    for (num = 0, i = 0; i < n_pts; i++)
    {
        /* select the interesting points */
        if (i + 1 < n_pts && y[i] == y[i + 1])
        {
            if (int_sw)
                z = odd_round(y[i], sw_odd);
            else
                z = atcg(y[i], sw);
            num = add_pair(y[i], z, num, old_v, new_v);
        }
    }

    /* (maybe) add Ymax, Ymin */
    set_bbox(n_pts, stik);
    if (int_sw)
    {
        num = add_pair(stik->lo_y, odd_round(stik->lo_y, sw_odd), num, old_v, new_v);
        num = add_pair(stik->hi_y, odd_round(stik->hi_y, sw_odd), num, old_v, new_v);
    }
    else
    {
        num = add_pair(stik->lo_y, atcg(stik->lo_y, sw), num, old_v, new_v);
        num = add_pair(stik->hi_y, atcg(stik->hi_y, sw), num, old_v, new_v);
    }

    /* now interpolate other points wrt these */
    sort_hints(num, old_v, new_v);
    for (i = 0; i < n_pts; i++)
    {
        y[i] = interp(y[i], num, old_v, new_v);
    }
    /* round contour endpoints to grid */

    for (i = 0, j = 0; i < stik->num; i++)
    {
        if (stik->type[i] == FS_MOVETO)
        {
            if (int_sw)
            {
                x[j] = odd_round(x[j], sw_odd);
                y[j] = odd_round(y[j], sw_odd);
            }
            else
            {
                x[j] = atcg(x[j], sw);
                y[j] = atcg(y[j], sw);
            }
            if (j > 0)
            {
                if (int_sw)
                {
                    x[j - 1] = odd_round(x[j - 1], sw_odd);
                    y[j - 1] = odd_round(y[j - 1], sw_odd);
                }
                else
                {
                    x[j - 1] = atcg(x[j - 1], sw);
                    y[j - 1] = atcg(y[j - 1], sw);
                }
            }
        }
        if (stik->type[i] == FS_QUADTO)
            j += 2;
        else
            j += 1;
    }

    /* round last point */
    if (int_sw)
    {
        x[j - 1] = odd_round(x[j - 1], sw_odd);
        y[j - 1] = odd_round(y[j - 1], sw_odd);
    }
    else
    {
        x[j - 1] = atcg(x[j - 1], sw);
        y[j - 1] = atcg(y[j - 1], sw);
    }
    set_bbox(n_pts, stik);
    /* cleanup and leave */
    FSS_free(_PS_ old_v);
    FSS_free(_PS_ new_v);
    return n_pts;
}


#define CROSS(A,B) (FixMul(A.x, B.y) - FixMul(A.y, B.x))
#define DOT(A,B) (FixMul(A.x,B.x) + FixMul(A.y,B.y))
/*#define COS_175_DEGREES -65287    not used*/
#define SIN_5_DEGREES 5712
#undef EXP_STIK_DBG

typedef struct
{
    int num_contours;
    int num_types, max_types;
    int num_points, max_points;
    FS_BYTE *types;
    FS_FIXED *x, *y;
} SECTION;

#ifdef EXP_STIK_DBG
void dump_section(char *s, SECTION *p)
{
    int i;
    double z = 65536.0;
    FS_FIXED *x = p->x;
    FS_FIXED *y = p->y;
    FS_BYTE *t = p->types;

    FS_PRINTF(("dump_section : %s\n", s));
    FS_PRINTF(("num_contours %d\n", p->num_contours));
    FS_PRINTF(("num_types %d max_types %d\n", p->num_types, p->max_types));
    FS_PRINTF(("num_points %d max_points %d\n", p->num_points, p->max_points));
    for (i = 0; i < p->num_types; i++)
    {
        switch (t[i])
        {
        case FS_MOVETO:
            FS_PRINTF(("%f %f moveto\n", x[0] / z, y[0] / z));
            x++;
            y++;
            break;
        case FS_LINETO:
            FS_PRINTF(("%f %f lineto\n", x[0] / z, y[0] / z));
            x++;
            y++;
            break;
        case FS_QUADTO:
            FS_PRINTF(("%f %f \n%f %f quadto\n", x[0] / z, y[0] / z, x[1] / z, y[1] / z));
            x += 2;
            y += 2;
            break;
        }
    }
    FS_PRINTF(("\n"));
    FS_FFLUSH(stdout);
}
#endif /* EXP_STIK_DBG */

static FS_ULONG init_section(_DS_ SECTION *p, int np, int nt)
{
    SYS_MEMSET(p, 0, sizeof(SECTION));
    p->max_types = nt;
    p->types = (FS_BYTE *)FSS_malloc(_PS_ nt);
    if (STATE.error) return STATE.error;
    p->max_points = np;
    p->x = (FS_FIXED *)FSS_malloc(_PS_ np * 4);
    if (STATE.error) return STATE.error;
    p->y = (FS_FIXED *)FSS_malloc(_PS_ np * 4);
    if (STATE.error) return STATE.error;
    return SUCCESS;
}

static FS_LONG add_M(_DS_ SECTION *p, FS_FIXED x, FS_FIXED y)
{
    p->num_contours++;
    if (p->num_types == p->max_types)
    {
        p->max_types += 32;
        p->types = (FS_BYTE *)FSS_realloc(_PS_ p->types, p->max_types);
        if (!p->types)
            return STATE.error;
    }
    if (p->num_points == p->max_points)
    {
        p->max_points += 64;
        p->x = (FS_FIXED *) FSS_realloc(_PS_ p->x, 4 * p->max_points);
        p->y = (FS_FIXED *) FSS_realloc(_PS_ p->y, 4 * p->max_points);
        if (!p->x || !p->y)
            return STATE.error;
    }
    p->types[p->num_types++] = FS_MOVETO;
    p->x[p->num_points] = x;
    p->y[p->num_points] = y;
    p->num_points++;
    return SUCCESS;
}

static FS_LONG add_L(_DS_ SECTION *p, FS_FIXED x, FS_FIXED y)
{
    if (p->num_types == p->max_types)
    {
        p->max_types += 32;
        p->types = (FS_BYTE *)FSS_realloc(_PS_ p->types, p->max_types);
        if (!p->types)
            return STATE.error;
    }
    if (p->num_points == p->max_points)
    {
        p->max_points += 64;
        p->x = (FS_FIXED *) FSS_realloc(_PS_ p->x, 4 * p->max_points);
        p->y = (FS_FIXED *) FSS_realloc(_PS_ p->y, 4 * p->max_points);
        if (!p->x || !p->y) return STATE.error;
    }
    p->types[p->num_types++] = FS_LINETO;
    p->x[p->num_points] = x;
    p->y[p->num_points] = y;
    p->num_points++;
    return SUCCESS;
}

static FS_LONG add_Q(_DS_ SECTION *p, FS_FIXED x1, FS_FIXED y1, FS_FIXED x2, FS_FIXED y2)
{
    if (p->num_types == p->max_types)
    {
        p->max_types += 32;
        p->types = (FS_BYTE *)FSS_realloc(_PS_ p->types, p->max_types);
        if (!p->types)
            return STATE.error;
    }
    if (p->num_points >= p->max_points - 1)
    {
        p->max_points += 64;
        p->x = (FS_FIXED *) FSS_realloc(_PS_ p->x, 4 * p->max_points);
        p->y = (FS_FIXED *) FSS_realloc(_PS_ p->y, 4 * p->max_points);
        if (!p->x || !p->y)
            return STATE.error;
    }
    p->types[p->num_types++] = FS_QUADTO;
    p->x[p->num_points] = x1;
    p->y[p->num_points] = y1;
    p->num_points++;
    p->x[p->num_points] = x2;
    p->y[p->num_points] = y2;
    p->num_points++;
    return SUCCESS;
}

/* append the elements of <right> to <left> BACKWARDS
*
* RIGHT
* 2.5   15        moveto
* 3     15
* 3     14.5    quadto
* 3     5.5        lineto
* <==> the segments
* quad(2.5 15) -> (3 15) -> (3 14.5)
* line(3 14.5) -> (3 5.5)
*
* LEFT
* 2.5   15        moveto
* 2     15
* 2     14.5    quadto
* 2     5.5        lineto
* 2     5
* 2.5   5        quadto
* 3     5
* 3     5.5        quadto
*
* want to reverse the segments of RIGHT, wrt
* both ORDER and DIRECTION.  In this case:
* line(3 5.5) -> (3 14.5)
* quad(3 14.5) -> (3 15) -> (2.5 15)
*
* or
*
* 3     14.5    lineto
* 3     15
* 2.5   15        quadto
*
* this can be done simply by processing the types in reverse
* (ignoring the initial moveto) and starting with the next
* to last <x,y> coordinate
*/
static FS_LONG copy_RL(_DS_ SECTION *right, SECTION *left)
{
    int i, j;
    FS_BYTE *types = right->types;
    FS_FIXED *x = right->x;
    FS_FIXED *y = right->y;

    for (i = right->num_types - 1, j = right->num_points - 2; i >= 1; i--)
    {
        if (types[i] == FS_QUADTO)
        {
            add_Q(_PS_ left, x[j], y[j], x[j - 1], y[j - 1]);
            if (STATE.error) return STATE.error;
            j -= 2;
        }
        else if (types[i] == FS_LINETO)
        {
            add_L(_PS_ left, x[j], y[j]);
            if (STATE.error) return STATE.error;
            j--;
        }
        else
        {
            add_M(_PS_ left, x[j], y[j]);
            if (STATE.error) return STATE.error;
            j--;
        }
    }
    right->num_points = 0;
    right->num_types = 0;
    return SUCCESS;
}

/* start a contour with either a 1/2 circle or a 1/4 square end */
static FS_LONG start_contour(_DS_ SECTION *left, SECTION *right, FS_FIXED x0, FS_FIXED y0,
                             FIXED_VECTOR *t1, FS_BOOLEAN round_end)
{
    FIXED_VECTOR n1;

    /* right hand normal */
    n1.y = -t1->x;
    n1.x = t1->y;

    if (round_end)
    {
        FS_FIXED sx, sy;

        /* start the contour with a half-round */
        sx = x0 - t1->x;
        sy = y0 - t1->y;

        /* right half */
        add_M(_PS_ right, sx, sy);
        if (STATE.error)
            return STATE.error;
        add_Q(_PS_ right, sx + n1.x, sy + n1.y, x0 + n1.x, y0 + n1.y);
        if (STATE.error)
            return STATE.error;

        /* left half */
        add_M(_PS_ left, sx, sy);  /* yes it is a duplicate... */
        if (STATE.error)
            return STATE.error;

        add_Q(_PS_ left, sx - n1.x, sy - n1.y, x0 - n1.x, y0 - n1.y);
        if (STATE.error)
            return STATE.error;
    }
    else
    {
        add_M(_PS_ right, x0 - n1.x - t1->x / 2, y0 - n1.y - t1->y / 2);
        if (STATE.error)
            return STATE.error;
        add_L(_PS_ right, x0 + n1.x - t1->x / 2, y0 + n1.y - t1->y / 2);
        if (STATE.error)
            return STATE.error;

        /* left half */
        add_M(_PS_ left, x0 - n1.x - t1->x / 2, y0 - n1.y - t1->y / 2);
        if (STATE.error)
            return STATE.error;
    }

    return STATE.error;
}

/* end the previous contour with either 1/2 circle or 1/4 square end */
static FS_LONG end_contour(_DS_ SECTION *left, SECTION *right, FS_FIXED sw, FS_BOOLEAN round_end)
{
    FIXED_VECTOR L;
    FIXED_VECTOR    R, M, ut1, t1, n1;
    int n;
    /* the previous contour ends with... */
    n = left->num_points - 1;
    L.x = left->x[n];
    L.y = left->y[n];

    if (round_end)
    {
        n = right->num_points - 1;
        R.x = right->x[n];
        R.y = right->y[n];

        /* a midpoint */
        M.x = (L.x + R.x) / 2;
        M.y = (L.y + R.y) / 2;

        /* unit vector from R to L */
        ut1.x = L.x - R.x;
        ut1.y = L.y - R.y;
        fixed_norm(&ut1);

        /* the right hand normal points in the direction
           of the continuation of the curve...  */
        t1.x = FixMul(sw / 2, ut1.x);
        t1.y = FixMul(sw / 2, ut1.y);
        n1.y = -t1.x;
        n1.x = t1.y;

        /* add the 1/2 circle to close the contour */
        add_Q(_PS_ right, R.x + n1.x, R.y + n1.y, M.x + n1.x, M.y + n1.y);
        if (STATE.error)
            return STATE.error;
        add_Q(_PS_ right, L.x + n1.x, L.y + n1.y, L.x, L.y);
        if (STATE.error)
            return STATE.error;
    }
    else
    {
        /* line to close the contour */
        add_L(_PS_ right, L.x, L.y);
        if (STATE.error)
            return STATE.error;
    }

    /* now copy all the points of <right> onto <left> */
    copy_RL(_PS_ right, left);

    return STATE.error;
}

/*lint -e801  Info -- Use of goto is deprecated */
/**************************************************************************/
/* function to create a regular closed outline from open-ended strokes    */
FS_OUTLINE *expand_stik(_DS_ FS_OUTLINE *stik, FS_LONG *_nt, FS_LONG *_np)
{
    FS_FIXED sw = STATE.stroke_pct * STATE.lpm;
    SECTION left, right;
    FIXED_VECTOR L, R, M, PR, PL, ut0, ut1, t1, n1, ut2, t2, n2, pt;
    FS_BOOLEAN round_end;
    FIXED_VECTOR h_t1;
    FS_FIXED x0, y0, x1, y1, x2, y2, xx0, yy0, xx2, yy2;
    int i, num;
    FS_OUTLINE *outl = 0;
    int nc, np, nt;
    FS_BYTE *types = (FS_BYTE *)(stik->type);
    FS_FIXED *x = stik->x;
    FS_FIXED *y = stik->y;
    int start = 1;    /* initialized to make the compiler happy */
    FS_BOOLEAN need_to_free_stik = 0;
    FS_FIXED sw_half;
    FS_FIXED xadj = 0;
    FS_FIXED yadj = 0;
    FS_FIXED diagonalsw;
    FS_FIXED orthogonalsw;
    FS_FIXED diagonalsw_half;
    FS_FIXED orthogonalsw_half;

#ifdef EXP_STIK_DBG
    dump_outline(_PS_ stik);
#endif

    /* round expanded stroke ends are used with bitmaps or */
    /* large sizes when the stroke width is > 3.5          */
    if (!(STATE.flags & FLAGS_GRAYSCALE) || (sw > 229376 /* 3.5 */))
        round_end = 1;
    else
        round_end = 0;

    /* integer stroke width is used with bitmaps or          */
    /* special raster effects                                */
    if (!(STATE.flags & FLAGS_GRAYSCALE) ||
        (STATE.flags & (FLAGS_EMBOSSED | FLAGS_ENGRAVED | FLAGS_OUTLINED | FLAGS_OUTLINED_2PIXEL | \
                        FLAGS_OUTLINED_UNFILLED)))
    {
        FS_FIXED fsw = MAX(FIXED_ONE, sw);
        sw = (fsw + FIXED_ONEHALF) & 0xFFFF0000;
        sw_half = sw / 2;
    }
    else
    {
        /* half stroke width adjusted for 26.6 resolution */
        sw_half = (FIXEDTODOT6(sw) / 2) << 10;
    }

    diagonalsw = orthogonalsw = sw;

    if (STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON)
    {
        if (sw < FIXED_ONE)
        {
            sw = FIXED_ONE;
            sw_half = FIXED_ONEHALF;
        }
        else
        {
            FS_FIXED fsw = (3277 * STATE.lpm + FIXED_ONEHALF) & 0xFFFF0000;

            if (fsw & FIXED_ONE)
            {
                /* this is the odd case */
                xadj = (sw - FIXED_ONE) / 2;
            }
            else
            {
                /* this is the even case */
                xadj = sw_half - FIXED_ONE;
            }

            if (xadj > FIXED_ONEHALF)
                xadj = xadj - FIXED_ONE;
            if (-xadj > FIXED_ONEHALF)
                xadj = xadj + FIXED_ONE;

            yadj = xadj;
        }
        diagonalsw = FixMul(sw, 55705);
        orthogonalsw = sw;
    }
    diagonalsw_half = diagonalsw / 2;
    orthogonalsw_half = orthogonalsw / 2;

    /* autohint non-offlined stroke fonts */
    if ( !(STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON))
    {
        FS_OUTLINE *ah_stik;
        /* stik passed in will be deleted by call chain */
        /* we need to free ah_stik when we are done with it */
        ah_stik = copy_outline(_PS_ stik, 0);
        if (ah_stik)
        {
            /* copy is successful */
            need_to_free_stik = 1;
            stik = ah_stik;
            types = stik->type;
            x = stik->x;
            y = stik->y;
        }
        autohint_stik(_PS_ stik, sw);
    }

    nc = stik->nc;

    *_np = *_nt = 0;    /* in preparation for failure ... */

    /* guess at parameters of expanded stik */
    np = (10 * nc) + (3 * stik->np) / 2;
    nt = (5 * nc) + (3 * stik->num) / 2;
    if (init_section(_PS_ & left, np, nt))
    {
        if (need_to_free_stik)
            FSS_free_char(_PS_ stik);
        return 0;
    }
    if (init_section(_PS_ & right, np, nt))
    {
        FSS_free(_PS_ left.x);
        FSS_free(_PS_ left.y);
        FSS_free(_PS_ left.types);
        if (need_to_free_stik)
            FSS_free_char(_PS_ stik);
        return 0;
    }

    num = stik->num;
    x0 = y0 = 0;    /* to remove a warning */
    PL.x = 0;
    PL.y = 0;
    PR.x = 0;
    PR.y = 0;
    ut0.x = 0;
    ut0.y = 0;
    for (i = 0; i < num; i++)
    {
        FS_FIXED sin, cos;

        switch (types[i])
        {
        case FS_MOVETO:
            x0 = *x++;
            y0 = *y++;
            x0 += xadj;
            y0 += yadj;

            {
                sw = orthogonalsw;
                sw_half = orthogonalsw_half;
            }

            if (right.num_points)
            {
                end_contour(_PS_ & left, &right, sw, round_end);
                if (STATE.error)
                    goto fail;

            }
            start = 1;
            break;

        case FS_LINETO:
case_lineto:
            x1 = *x++;
            y1 = *y++;
            x1 += xadj;
            y1 += yadj;

            if (x1 == x0 || y1 == y0)
            {
                sw = orthogonalsw;
                sw_half = orthogonalsw_half;
            }
            else
            {
                sw = diagonalsw;
                sw_half = diagonalsw_half;
            }


            /* degenerate case, such as a period */
            if (x0 == x1 && y0 == y1)
            {
                if ((i > 0) && (types[i - 1] != FS_MOVETO))
                    continue; /* skip continuing line case */


                if (round_end)
                {
                    /* right half */
                    add_M(_PS_ & right, x0 + sw_half, y0);
                    if (STATE.error)
                        goto fail;
                    add_Q(_PS_ & right, x0 + sw_half, y0 + sw_half, x0, y0 + sw_half);
                    if (STATE.error)
                        goto fail;
                    add_Q(_PS_ & right, x0 - sw_half, y0 + sw_half, x0 - sw_half, y0);
                    if (STATE.error)
                        goto fail;

                    /* left half */
                    add_M(_PS_ & left, x0 - sw_half, y0);
                    if (STATE.error)
                        goto fail;
                    add_Q(_PS_ & left, x0 - sw_half, y0 - sw_half, x0, y0 - sw_half);
                    if (STATE.error)
                        goto fail;
                    add_Q(_PS_ & left, x0 + sw_half, y0 - sw_half, x0 + sw_half, y0);
                    if (STATE.error)
                        goto fail;
                }
                else
                {
                    /* add a square */
                    add_M(_PS_ & right, x0 + sw_half, y0 - sw_half);
                    if (STATE.error)
                        goto fail;

                    add_L(_PS_ & right, x0 + sw_half, y0 + sw_half);
                    if (STATE.error)
                        goto fail;

                    add_L(_PS_ & right, x0 - sw_half, y0 + sw_half);
                    if (STATE.error)
                        goto fail;

                    add_L(_PS_ & right, x0 - sw_half, y0 - sw_half);
                    if (STATE.error)
                        goto fail;

                    add_M(_PS_ & left, x0 - sw_half, y0 - sw_half);
                    if (STATE.error)
                        goto fail;

                    add_L(_PS_ & left, x0 + sw_half, y0 - sw_half);
                    if (STATE.error)
                        goto fail;
                }

                /* save end point and tangent */
                /* unit tangent in direction of line */
                ut1.x = x1 - x0;
                ut1.y = y1 - y0;
                fixed_norm(&ut1);

                x0 = x1;
                y0 = y1;
                ut0 = ut1;
                break;
            }

            /* unit tangent in direction of line */
            ut1.x = x1 - x0;
            ut1.y = y1 - y0;
            fixed_norm(&ut1);


            /* tangent and normal of length sw/2 */
            t1.x = FixMul(sw_half, ut1.x);
            t1.y = FixMul(sw_half, ut1.y);
            n1.y = -t1.x;
            n1.x = t1.y;

            if (start)
            {
                start_contour(_PS_ & left, &right, x0, y0, &t1, round_end);
                if (STATE.error)  goto fail;

                /* it IS a smooth joint */
                ut0 = ut1;
                /* no longer the start of a contour */
                start = 0;
            }

            /* ? is the joint angular -- more than 5 degrees */
            sin = CROSS(ut0, ut1);
            cos = DOT(ut0, ut1);

            if (cos < 0) /* a sharp turn, more than 90 degrees    */
            {
                FIXED_VECTOR t0, n0;

                t0.x = FixMul(sw_half, ut0.x);
                t0.y = FixMul(sw_half, ut0.y);
                n0.y = -t0.x;
                n0.x = t0.y;
                L.x = x0 + t0.x - n0.x;
                L.y = y0 + t0.y - n0.y;
                R.x = x0 - t1.x - n1.x;
                R.y = y0 - t1.y - n1.y;
                M.x = (L.x + R.x) / 2;
                M.y = (L.y + R.y) / 2;
                add_Q(_PS_ & left, L.x, L.y, M.x, M.y);
                add_Q(_PS_ & left, R.x, R.y, R.x + t1.x, R.y + t1.y);
                add_L(_PS_ & right, x0 + n1.x, y0 + n1.y);
            }
            else if (sin > SIN_5_DEGREES)
            {
                /* a left turn, causes gap on right */
                L.x = x0 - n1.x;
                L.y = y0 - n1.y;
                R.x = x0 + n1.x;
                R.y = y0 + n1.y;

                /* add the point on the left
                    !!! could cause an interior triangle !!! */
                add_L(_PS_ & left, L.x, L.y);
                if (STATE.error)
                    goto fail;

                /* find the intersection point of the right-hand
                    vectors (PR + ut0) and (R + ut1) */
                intersect(PR.x, PR.y, &ut0, R.x, R.y, &ut1, &pt);

                /* span the gap with a little parabola */
                add_Q(_PS_ & right, pt.x, pt.y, R.x, R.y);
                if (STATE.error)
                    goto fail;
            }
            else if (sin < -SIN_5_DEGREES)
            {
                /* a right turn, causes gap on left */
                L.x = x0 - n1.x;
                L.y = y0 - n1.y;
                R.x = x0 + n1.x;
                R.y = y0 + n1.y;

                /* find the intersection point of the left-hand
                   vectors (PL + ut0) and (L + ut1) */
                intersect(PL.x, PL.y, &ut0, L.x, L.y, &ut1, &pt);
                /* span the gap with a little parabola */
                add_Q(_PS_ & left, pt.x, pt.y, L.x, L.y);
                if (STATE.error)
                    goto fail;

                /* add the point on the right
                   !!! could cause a interior triangle !!! */
                add_L(_PS_ & right, R.x, R.y);
                if (STATE.error)
                    goto fail;
            }

            /* now that gaps are filled ... do the left/right lines */
            if (round_end)
            {
                add_L(_PS_ & right, PR.x = x1 + n1.x, PR.y = y1 + n1.y);
                if (STATE.error)
                    goto fail;
                add_L(_PS_ & left, PL.x = x1 - n1.x, PL.y = y1 - n1.y);
                if (STATE.error)
                    goto fail;
            }
            else
            {
                h_t1.x = FixMul(sw / 4, ut1.x);
                h_t1.y = FixMul(sw / 4, ut1.y);
                add_L(_PS_ & right, PR.x = x1 + n1.x + h_t1.x, PR.y = y1 + n1.y + h_t1.y);
                if (STATE.error)
                    goto fail;
                add_L(_PS_ & left, PL.x = x1 - n1.x + h_t1.x, PL.y = y1 - n1.y + h_t1.y);
                if (STATE.error)
                    goto fail;
            }

            /* save end point and tangent */
            x0 = x1;
            y0 = y1;
            ut0 = ut1;
            break;

        case FS_QUADTO:
            x1 = *x++;
            y1 = *y++;
            x2 = *x++;
            y2 = *y++;
            x1 += xadj;
            y1 += yadj;
            x2 += xadj;
            y2 += yadj;

            {
                sw = diagonalsw;
                sw_half = diagonalsw_half;
            }


            /* degenerate cases */
            if ((x0 == x1 && y0 == y1) || (x1 == x2 && y1 == y2))
            {
                *--x = x2;
                *--y = y2;
                goto case_lineto;
            }

            /* unit tangent at <x0,y0> */
            ut1.x = x1 - x0;
            ut1.y = y1 - y0;
            fixed_norm(&ut1);

            /* tangent and normal of length sw/2 */
            t1.x = FixMul(sw_half, ut1.x);
            t1.y = FixMul(sw_half, ut1.y);
            n1.y = -t1.x;
            n1.x = t1.y;

            /* unit tangent at <x2,y2> */
            ut2.x = x2 - x1;
            ut2.y = y2 - y1;
            fixed_norm(&ut2);

            /* check and fix poorly hinted cases. sin is the angle between the legs */
            /* of the quad. if almost flat ... use a line instead                   */
            sin = CROSS(ut1, ut2);
            if ( ABS(sin) < SIN_30_DEGREES &&
                    ((ut1.x & 0x80000000) != (ut2.x & 0x80000000)) &&
                    ((ut1.y & 0x80000000) != (ut2.y & 0x80000000)) )
            {
                *--x = x2;
                *--y = y2;
                goto case_lineto;
            }

            /* tangent and normal of length sw/2 */
            t2.x = FixMul(sw_half, ut2.x);
            t2.y = FixMul(sw_half, ut2.y);
            n2.y = -t2.x;
            n2.x = t2.y;

            if (start)
            {
                start_contour(_PS_ & left, &right, x0, y0, &t1, round_end);
                if (STATE.error)  goto fail;

                /* it IS a smooth joint */
                ut0 = ut1;

                /* no longer the start of a contour */
                start = 0;
            }

            /* ? is the joint angular -- more than 5 degrees */
            sin = CROSS(ut0, ut1);
            cos = DOT(ut0, ut1);
            if (cos < 0)
            {
                FIXED_VECTOR t0, n0;

                t0.x = FixMul(sw_half, ut0.x);
                t0.y = FixMul(sw_half, ut0.y);
                n0.y = -t0.x;
                n0.x = t0.y;
                L.x = x0 + t0.x - n0.x;
                L.y = y0 + t0.y - n0.y;
                R.x = x0 - t1.x - n1.x;
                R.y = y0 - t1.y - n1.y;
                M.x = (L.x + R.x) / 2;
                M.y = (L.y + R.y) / 2;
                add_Q(_PS_ & left, L.x, L.y, M.x, M.y);
                add_Q(_PS_ & left, R.x, R.y, R.x + t1.x, R.y + t1.y);
                add_L(_PS_ & right, x0 + n1.x, y0 + n1.y);
            }
            else if (sin > SIN_5_DEGREES)
            {
                /* a left turn, causes gap on right */
                L.x = x0 - n1.x;
                L.y = y0 - n1.y;
                R.x = x0 + n1.x;
                R.y = y0 + n1.y;

                /* add the point on the left
                 !!! could cause an interior triangle !!! */
                add_L(_PS_ & left, L.x, L.y);
                if (STATE.error) goto fail;

                /* find the intersection point of the right-hand
                   vectors (PR + ut0) and (R + ut1) */
                intersect(PR.x, PR.y, &ut0, R.x, R.y, &ut1, &pt);

                /* span the gap with a little parabola */
                add_Q(_PS_ & right, pt.x, pt.y, R.x, R.y);
                if (STATE.error) goto fail;
            }
            else if (sin < -SIN_5_DEGREES)
            {
                /* a right turn, causes gap on left */
                L.x = x0 - n1.x;
                L.y = y0 - n1.y;
                R.x = x0 + n1.x;
                R.y = y0 + n1.y;

                /* find the intersection point of the left-hand
                   vectors (PL + ut0) and (L + ut1) */
                intersect(PL.x, PL.y, &ut0, L.x, L.y, &ut1, &pt);

                /* span the gap with a little parabola */
                add_Q(_PS_ & left, pt.x, pt.y, L.x, L.y);
                if (STATE.error) goto fail;

                /* add the point on the right
                   !!! could cause a interior triangle !!! */
                add_L(_PS_ & right, R.x, R.y);
                if (STATE.error)
                    goto fail;
            }

            /* now that any gaps have been filled, do the left/right quads
               start point of the new left parabola */
            xx0 = x0 - n1.x;
            yy0 = y0 - n1.y;
            /* endpoint of the new left parabola */
            xx2 = x2 - n2.x;
            yy2 = y2 - n2.y;
            /* control point is at intersection of control arms */
            intersect(xx0, yy0, &t1, xx2, yy2, &t2, &pt);

            /* add the left quad */
            add_Q(_PS_ & left, pt.x, pt.y, PL.x = xx2, PL.y = yy2);
            if (STATE.error)
                goto fail;

            /* start point of the new right parabola */
            xx0 = x0 + n1.x;
            yy0 = y0 + n1.y;
            /* end point of the new right parabola */
            xx2 = x2 + n2.x;
            yy2 = y2 + n2.y;
            /* control point is at intersection of control arms */
            intersect(xx0, yy0, &t1, xx2, yy2, &t2, &pt);

            /* add the right quad */
            add_Q(_PS_ & right, pt.x, pt.y, PR.x = xx2, PR.y = yy2);
            if (STATE.error) goto fail;

            /* save final point and tangent */
            x0 = x2;
            y0 = y2;
            ut0 = ut2;

            break;

        default:
            /* there is no default. */
            goto fail;
        } /* switch (types[i]) */

    } /* for (i=0; i<num; i++) */


    /* finish the last contour */
    if (right.num_points)
    {
        end_contour(_PS_ & left, &right, sw, round_end);
        if (STATE.error)
            goto fail;
    }

    /* almost done ... now make the mess into an FS_OUTLINE */
    outl = setup_outline(_PS_ (FS_SHORT)left.num_contours, left.num_types, left.num_points);
    if (outl)
    {
        FS_BYTE *t = (FS_BYTE *)(outl->type);
        FS_FIXED *px = (FS_FIXED *)(outl->x);
        FS_FIXED *py = (FS_FIXED *)(outl->y);

        /* move the data into an FS_OUTLINE */
        if (t)
        {
            SYS_MEMCPY(t, left.types, left.num_types);
            /* it is now an outline char */
            outl->type[0] |= OUTLINE_CHAR;
        }
        if (px) SYS_MEMCPY(px, left.x, left.num_points * 4);
        if (py) SYS_MEMCPY(py, left.y, left.num_points * 4);
        set_bbox(left.num_points, outl);
        *_np = left.num_points;
        *_nt = left.num_types;
        outl->i_dx = stik->i_dx;
        outl->i_dy = stik->i_dy;
        outl->dx = stik->dx;
        outl->dy = stik->dy;
    }

fail:

    /* free temps */
    FSS_free(_PS_ right.x);
    FSS_free(_PS_ right.y);
    FSS_free(_PS_ right.types);
    FSS_free(_PS_ left.x);
    FSS_free(_PS_ left.y);
    FSS_free(_PS_ left.types);
    if (need_to_free_stik)
        FSS_free_char(_PS_ stik);
    return outl;

}
/*lint +e801  Info -- Use of goto is deprecated */

static FS_VOID degenerate_quad1(FS_BYTE *ot, FS_FIXED *ox, FS_FIXED *oy,
                                FS_FIXED stroke_half, FS_FIXED x0,
                                FS_FIXED y0)
{
    *ot++ = FS_MOVETO;
    *ox++ = x0 - stroke_half;
    *oy++ = y0;

    /* must be CLOCKWISE */
    *ot++ = FS_QUADTO;
    *ox++ = x0 - stroke_half;
    *oy++ = y0 + stroke_half;
    *ox++ = x0;
    *oy++ = y0 + stroke_half;
    *ot++ = FS_QUADTO;
    *ox++ = x0 + stroke_half;
    *oy++ = y0 + stroke_half;
    *ox++ = x0 + stroke_half;
    *oy++ = y0;
    *ot++ = FS_QUADTO;
    *ox++ = x0 + stroke_half;
    *oy++ = y0 - stroke_half;
    *ox++ = x0;
    *oy++ = y0 - stroke_half;
    *ot++ = FS_QUADTO;
    *ox++ = x0 - stroke_half;
    *oy++ = y0 - stroke_half;
    *ox   = x0 - stroke_half;
    *oy   = y0;
}


static FS_VOID degenerate_quad2(FS_BYTE *ot, FS_FIXED *ox, FS_FIXED *oy,
                                FS_FIXED stroke, FS_FIXED x0, FS_FIXED x1,
                                FS_FIXED y0, FS_FIXED y1)
{

    FIXED_VECTOR p0, n0;
    /* unit tangent at <x0,y0> */
    p0.x = x1 - x0;
    p0.y = y1 - y0;
    fixed_norm(&p0);

    if (ABS(p0.x) == MYINFINITY)
        p0.x = 0;
    if (ABS(p0.y) == MYINFINITY)
        p0.y = 0;

    /* tangent and normal stretched to length == stroke/2 */
    p0.x = FixMul(p0.x, stroke) >> 1;
    p0.y = FixMul(p0.y, stroke) >> 1;
    n0.x = -p0.y;
    n0.y = p0.x;

    /* assemble the envelope of the line stick */
    *ot++ = FS_MOVETO;    /* left of <x0,y0> */
    *ox++ = x0 + n0.x;
    *oy++ = y0 + n0.y;

    *ot++ = FS_LINETO;    /* to left of <x1,y1> */
    *ox++ = x1 + n0.x;
    *oy++ = y1 + n0.y;

    *ot++ = FS_QUADTO;    /* first half of round at <x1,y1> */
    *ox++ = x1 + n0.x + p0.x;
    *oy++ = y1 + n0.y + p0.y;
    *ox++ = x1 + p0.x;
    *oy++ = y1 + p0.y;

    *ot++ = FS_QUADTO;    /* second half of round at <x1,y1> */
    *ox++ = x1 + p0.x - n0.x;
    *oy++ = y1 + p0.y - n0.y;
    *ox++ = x1 - n0.x;
    *oy++ = y1 - n0.y;

    *ot++ = FS_LINETO;    /* to right of <x0,y0> */
    *ox++ = x0 - n0.x;
    *oy++ = y0 - n0.y;

    *ot++ = FS_QUADTO;    /* first half of round at <x0,y0> */
    *ox++ = x0 - (p0.x + n0.x);
    *oy++ = y0 - (p0.y + n0.y);
    *ox++ = x0 - p0.x;
    *oy++ = y0 - p0.y;

    *ot   = FS_QUADTO;    /* second half of round at <x0,y0> */
    *ox++ = x0 + n0.x - p0.x;
    *oy++ = y0 + n0.y - p0.y;
    *ox   = x0 + n0.x;
    *oy   = y0 + n0.y;    /* back to start point */
    return;

}
/****************************************************************/
/* round/round expansion of a stik character used for bitmap    */
/* rendering of non-offlined stroke fonts                       */
/* stik passed in will be deleted by caller chain (make_bitmap) */
FS_OUTLINE *bitmap_expand_stik(_DS_ FS_OUTLINE *stik, FS_LONG *_nt, FS_LONG *_np)
{
    FS_ULONG size;
    FS_LONG n_points;
    FS_SHORT i, n_types, num;
    FS_ULONG type_off, x_off, y_off;
    FS_BYTE *p;
    FS_OUTLINE *outl;
    FS_BYTE *st, *ot;
    FS_FIXED *sx, *sy, *ox, *oy;
    FS_FIXED fsw, stroke;
    FS_FIXED xm, ym, x0 = 0, y0 = 0, x1, y1, x2, y2; /* misc */
    FS_FIXED stroke_half;
    FS_BOOLEAN need_to_free_stik = 0;

    num = stik->num;
    st = (FS_BYTE *)(stik->type);
    sx = (FS_FIXED *)(stik->x);
    sy = (FS_FIXED *)(stik->y);

    /* the 'ideal' stroke width */
    fsw = MAX(FIXED_ONE, STATE.stroke_pct * STATE.lpm);

    stroke = (fsw + FIXED_ONEHALF) & 0xFFFF0000;

    if ( !(STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON))
    {
        FS_OUTLINE *ah_stik;
        /* stik passed in will be deleted by call chain */
        /* we need to free ah_stik when we are done with it */
        ah_stik = copy_outline(_PS_ stik, 0);
        if (ah_stik)
        {
            /* copy is successful */
            need_to_free_stik = 1;
            stik = ah_stik;
            st = (FS_BYTE *)(stik->type);
            sx = (FS_FIXED *)(stik->x);
            sy = (FS_FIXED *)(stik->y);
        }
        autohint_stik(_PS_ stik, (FS_FIXED)(STATE.stroke_pct * STATE.lpm));
    }

    /* how much room will the expanded stik occupy  -- actually
    * it might be smaller than this calculation suggests, due
    * to any degenerate quadratics which become line segments */
    n_types = 0;
    n_points = 0;
    for (i = 0; i < num; i++)
    {
        if (st[i] == FS_LINETO)
        {
            n_points += 11;
            n_types += 7;
        }
        else if (st[i] == FS_QUADTO)
        {
            n_points += 17;
            n_types += 9;
        }
    }

    /* allocate the new outline structure */
    size = sizeof(FS_OUTLINE);
    FS_ALIGN(size);
    type_off = size;
    size += n_types;
    FS_ALIGN(size);
    x_off = size;
    size += (4 * n_points);
    FS_ALIGN(size);
    y_off = size;
    size += (4 * n_points);

#ifdef FS_MEM_DBG
    STATE.memdbgid = "FS_OUTLINE";
#endif
    p = (FS_BYTE *)FSS_calloc(_PS_ size);
    if (p == NULL)  /* Coverity does not get along with "if (STATE.error)" */
    {
        if (need_to_free_stik) FSS_free_char(_PS_ stik);
        return 0;
    }
    outl = (FS_OUTLINE *)p;
    outl->size = size;
    outl->num = 0;
    outl->type = (FS_BYTE *)(p + type_off);
    outl->x = (FS_FIXED *)(p + x_off);
    outl->y = (FS_FIXED *)(p + y_off);

    /* generate the sausages from the stik */
    ox = outl->x;
    oy = outl->y;
    ot = outl->type;
    n_points = 0;

    stroke_half = stroke >> 1;
    for (i = 0; i < num; i++)
    {
        FIXED_VECTOR p0, n0;       /* parallel and normal at quad start point */
        FIXED_VECTOR pm, nm;       /* parallel and normal at quad midpoint */
        FIXED_VECTOR p2, n2;       /* parallel and normal at quad endpoint */
        FIXED_VECTOR c1, c2, c3, c4; /* control points */
        FS_FIXED x[6], y[6];           /* useful points on quadratics */

        switch (st[i] & 0x7F)
        {
        case FS_MOVETO:
            x0 = *sx++;
            y0 = *sy++;
            break;


        case FS_LINETO:
            x1 = *sx++;
            y1 = *sy++;

            /* degenerate line ? */

            if (x0 == x1 && y0 == y1)
            {
                outl->num += 5;
                n_points += 9;
                degenerate_quad1(ot, ox, oy, stroke_half, x0, y0);
                ot += 5;
                ox += 9;
                oy += 9;
            }
            else
            {
                outl->num += 7;
                n_points += 11;
                degenerate_quad2(ot, ox, oy, stroke, x0, x1, y0, y1);
                ot += 7;
                ox += 11;
                oy += 11;
            }
            x0 = x1;
            y0 = y1;
            break;

        case FS_QUADTO:
            x1 = *sx++;
            y1 = *sy++;
            x2 = *sx++;
            y2 = *sy++;

            /* hmmm */
            if (x0 == x1 && y0 == y1 && x1 == x2 && y1 == y2)
            {
                x0 = x2;
                y0 = y2;
                break;
            }

            /* use a line <x0,y0> -> <x2,y2> */
            if (x0 == x1 && y0 == y1)
            {
                x1 = x2;
                y1 = y2;
                if (x0 == x1 && y0 == y1)
                {
                    outl->num += 5;
                    n_points += 9;
                    degenerate_quad1(ot, ox, oy, stroke_half, x0, y0);
                    ot += 5;
                    ox += 9;
                    oy += 9;
                }
                else
                {
                    outl->num += 7;
                    n_points += 11;
                    degenerate_quad2(ot, ox, oy, stroke, x0, x1, y0, y1);
                    ot += 7;
                    ox += 11;
                    oy += 11;
                }
                x0 = x1;
                y0 = y1;
                break;
            }

            /* use a line <x0,y0> -> <x1,y1> */
            if (x1 == x2 && y1 == y2)
            {
                if (x0 == x1 && y0 == y1)
                {
                    outl->num += 5;
                    n_points += 9;
                    degenerate_quad1(ot, ox, oy, stroke_half, x0, y0);
                    ot += 5;
                    ox += 9;
                    oy += 9;
                }
                else
                {
                    outl->num += 7;
                    n_points += 11;
                    degenerate_quad2(ot, ox, oy, stroke, x0, x1, y0, y1);
                    ot += 7;
                    ox += 11;
                    oy += 11;
                }
                x0 = x1;
                y0 = y1;
                break;
            }

            /* unit tangent at <x0,y0> */
            p0.x = x1 - x0;
            p0.y = y1 - y0;
            fixed_norm(&p0);

            /* unit tangent at <x2,y2> */
            p2.x = x2 - x1;
            p2.y = y2 - y1;
            fixed_norm(&p2);

            /* In very rare cases division by zero returns  MYINFINITY */
            /* That causes problem in determining the angle between two lines */
            /*   So we opt  for degenerate quad */

            if ((ABS(p0.x) == MYINFINITY) || (ABS(p0.y) == MYINFINITY) ||
                (ABS(p2.x) == MYINFINITY) || (ABS(p2.y) == MYINFINITY))
            {
                xm = 0;
            }
            else
                /* <xm> is the sin of the included angle between the legs
                 * of the quadratic. if almost flat ... use a line instead */
                xm = FixMul(p0.x, p2.y) - FixMul(p2.x, p0.y);

            if (ABS(xm) < SIN_2_DEGREES)
            {
                x1 = x2;
                y1 = y2;
                if (x0 == x1 && y0 == y1)
                {
                    outl->num += 5;
                    n_points += 9;
                    degenerate_quad1(ot, ox, oy, stroke_half, x0, y0);
                    ot += 5;
                    ox += 9;
                    oy += 9;
                }
                else
                {
                    outl->num += 7;
                    n_points += 11;
                    degenerate_quad2(ot, ox, oy, stroke, x0, x1, y0, y1);
                    ot += 7;
                    ox += 11;
                    oy += 11;
                }
                x0 = x1;
                y0 = y1;
                break;
            }

            /* fix poorly delta hinted control points */
            if ( ABS(xm) < SIN_30_DEGREES &&
                 ((p0.x & 0x80000000) != (p2.x & 0x80000000)) &&
                 ((p0.y & 0x80000000) != (p2.y & 0x80000000)) )
            {
                outl->num += 7;
                n_points += 11;
                degenerate_quad2(ot, ox, oy, stroke, x0, x2, y0, y2);
                ot += 7;
                ox += 11;
                oy += 11;

                x0 = x2;
                y0 = y2;
                break;
            }

            /* now that we know we're really gonna be here ... */
            outl->num += 9;
            n_points += 17;

            /* parametric midpoint and tangent at midpoint */
            xm = (x0 >> 2) + (x1 >> 1) + (x2 >> 2);
            ym = (y0 >> 2) + (y1 >> 1) + (y2 >> 2);
            pm.x = (x1 + x2) / 2 - (x1 + x0) / 2;
            pm.y = (y1 + y2) / 2 - (y1 + y0) / 2;
            fixed_norm(&pm);

            if (ABS(pm.x) == MYINFINITY)
                pm.x = 0;
            if (ABS(pm.y) == MYINFINITY)
                pm.y = 0;
            /* make normal offset vectors of length == stroke/2
             * but we still need UNIT tangent vectors at this point */
            n0.x = -FixMul(p0.y, stroke) >> 1;
            n0.y = FixMul(p0.x, stroke) >> 1;
            nm.x = -FixMul(pm.y, stroke) >> 1;
            nm.y = FixMul(pm.x, stroke) >> 1;
            n2.x = -FixMul(p2.y, stroke) >> 1;
            n2.y = FixMul(p2.x, stroke) >> 1;

            /* The left side envelope of a single quadratic is
             * a pair of quadratics.  The first starts left of
             * <x0,y0> and ends left of <xm,ym>.  The second
             * starts left of <xm,ym> and ends left of <x2,y2>.
             * Then a second pair of quads wraps the right side.
             *
             * We know the points and slopes at these points which
             * allows us to calculate the control points which lie
             * at the intersection of these segments for the 2 quads.
             *
             * So we calculate the endpoints of these quadratics
             * (saving in a temp array) and infer the control points */
            x[0] = x0 + n0.x;
            y[0] = y0 + n0.y;
            x[1] = xm + nm.x;
            y[1] = ym + nm.y;
            intersect(x[0], y[0], &p0, x[1], y[1], &pm, &c1);
            x[2] = x2 + n2.x;
            y[2] = y2 + n2.y;
            intersect(x[1], y[1], &pm, x[2], y[2], &p2, &c2);
            x[3] = x2 - n2.x;
            y[3] = y2 - n2.y;
            x[4] = xm - nm.x;
            y[4] = ym - nm.y;
            intersect(x[3], y[3], &p2, x[4], y[4], &pm, &c3);
            x[5] = x0 - n0.x;
            y[5] = y0 - n0.y;
            intersect(x[4], y[4], &pm, x[5], y[5], &p0, &c4);

            /* now we need the scaled tangent vectors -- to
             * calculate the coords of the projecting ends */
            p0.x = n0.y;
            p0.y = -n0.x;
            pm.x = nm.y;
            pm.y = -nm.x;
            p2.x = n2.y;
            p2.y = -n2.x;

            /* assemble the envelope of the quadratic stick */

            *ot++ = FS_MOVETO;    /* left of quadratic start point */
            *ox++ = x[0];
            *oy++ = y[0];

            *ot++ = FS_QUADTO;    /* left side - first half */
            *ox++ = c1.x;
            *oy++ = c1.y;
            *ox++ = x[1];
            *oy++ = y[1];

            *ot++ = FS_QUADTO;    /* left side - second half */
            *ox++ = c2.x;
            *oy++ = c2.y;
            *ox++ = x[2];
            *oy++ = y[2];

            *ot++ = FS_QUADTO;    /* far round - first half */
            *ox++ = x2 + n2.x + p2.x;
            *oy++ = y2 + n2.y + p2.y;
            *ox++ = x2 + p2.x;
            *oy++ = y2 + p2.y;

            *ot++ = FS_QUADTO;    /* far round - second half */
            *ox++ = x2 + p2.x - n2.x;
            *oy++ = y2 + p2.y - n2.y;
            *ox++ = x2 - n2.x;
            *oy++ = y2 - n2.y;

            *ot++ = FS_QUADTO;    /* right side - first half */
            *ox++ = c3.x;
            *oy++ = c3.y;
            *ox++ = x[4];
            *oy++ = y[4];

            *ot++ = FS_QUADTO;    /* right side - second half */
            *ox++ = c4.x;
            *oy++ = c4.y;
            *ox++ = x[5];
            *oy++ = y[5];

            *ot++ = FS_QUADTO;    /* near round - first half */
            *ox++ = x0 - (n0.x + p0.x);
            *oy++ = y0 - (n0.y + p0.y);
            *ox++ = x0 - p0.x;
            *oy++ = y0 - p0.y;

            *ot++ = FS_QUADTO;    /* near round - second half */
            *ox++ = x0 + n0.x - p0.x;
            *oy++ = y0 + n0.y - p0.y;
            *ox++ = x[0];
            *oy++ = y[0];     /* back to start point */

            x0 = x2;
            y0 = y2;
            break;
        default:
            break;
        }
    }

    /* compute the bounding box */
    ox = outl->x;
    oy = outl->y;
    outl->lo_x = outl->hi_x = ox[0];
    outl->lo_y = outl->hi_y = oy[0];
    for (i = 1; i < n_points; i++)
    {
        if (ox[i] < outl->lo_x) outl->lo_x = ox[i];
        if (ox[i] > outl->hi_x) outl->hi_x = ox[i];
        if (oy[i] < outl->lo_y) outl->lo_y = oy[i];
        if (oy[i] > outl->hi_y) outl->hi_y = oy[i];
    }

    /* and set the escapements */
    outl->i_dx = stik->i_dx;
    outl->i_dy = stik->i_dy;
    outl->dx = stik->dx;
    outl->dy = stik->dy;


    /* it is now an outline char */
    outl->type[0] |= OUTLINE_CHAR;

    /* record num_types and num_points ... maybe */
    if (_np) (*_np) = n_points;
    if (_nt) (*_nt) = outl->num;

    if (need_to_free_stik) FSS_free_char(_PS_ stik);
    return outl;
}


/****************************************************************/
/************ draw thin sticks direct to a bitmap ***************/
/****************************************************************/

/*
* set the pixels corresponding to the multi-bit pen with diameter
* of the current stroke_width, "centered" at the argument coord.
*/

static FS_VOID set_pixel(_DS_ int x, int y)
{
    FS_BITMAP *bmap = STATE.server->bmap;
    FS_USHORT col = (FS_USHORT)x - bmap->lo_x;
    FS_USHORT row = (FS_USHORT)(bmap->hi_y - y);
    FS_SHORT sw = STATE.cur_sfnt->senv->stroke_width;
    FS_BYTE *p;

    /* byte address of proper row */
    p = bmap->bits + (row * bmap->bpl);

    if (col >= bmap->width || row >= bmap->height)
        return;

    if (sw == 1)
        p[col >> 3] |= fs_mask[col & 7];
    else if (sw == 2)
    {
        /* every pixel whose center is within a radius of 1 ... offset by 1/2 pixel
        *  XO
        *  XX
        */

        unsigned int c1, c2;
        FS_BYTE b1, b2;

        if (col - 1 < 0 || row + 1 >= bmap->height)
            return;

        c1 = (FS_USHORT)(col - 1) >> 3;
        b1 = fs_mask[(col - 1) & 7];
        c2 = (FS_USHORT)(col + 0) >> 3;
        b2 = fs_mask[(col + 0) & 7];

        /* proper row */
        p[c1] |= b1;
        p[c2] |= b2;

        /* next row */
        p += bmap->bpl;
        p[c1] |= b1;
        p[c2] |= b2;

    }

    else if (sw == 3)
    {
        /* every pixel whose center is within a radius of 1.5 (corners are 1.414)
        *    XXX
        *    XOX
        *    XXX
        */
        unsigned int c1, c2, c3;
        FS_BYTE b1, b2, b3;

        if (col - 1 < 0 || col + 1 >= bmap->width || row - 1 < 0 || row + 1 >= bmap->height)
            return;

        c1 = (FS_USHORT)(col - 1) >> 3;
        b1 = fs_mask[(col - 1) & 7];
        c2 = (FS_USHORT)(col + 0) >> 3;
        b2 = fs_mask[(col + 0) & 7];
        c3 = (FS_USHORT)(col + 1) >> 3;
        b3 = fs_mask[(col + 1) & 7];

        /* prev row */
        p -= bmap->bpl;
        p[c1] |= b1;
        p[c2] |= b2;
        p[c3] |= b3;

        /* proper row */
        p += bmap->bpl;
        p[c1] |= b1;
        p[c2] |= b2;
        p[c3] |= b3;

        /* next row */
        p += bmap->bpl;
        p[c1] |= b1;
        p[c2] |= b2;
        p[c3] |= b3;
    }
    else if (sw == 4)
    {
        /* set this pattern -- 'O' is the argument pixel
        *      XX
        *     XXOX
        *     XXXX
        *      XX
        */
        unsigned int c1, c2, c3, c4;
        FS_BYTE b1, b2, b3, b4;

        if (col - 2 < 0 || col + 1 >= bmap->width || row - 1 < 0 || row + 2 >= bmap->height)
            return;

        c1 = (FS_USHORT)(col - 2) >> 3;
        b1 = fs_mask[(col - 2) & 7];
        c2 = (FS_USHORT)(col - 1) >> 3;
        b2 = fs_mask[(col - 1) & 7];
        c3 = (col + 0) >> 3;
        b3 = fs_mask[(col + 0) & 7];
        c4 = (col + 1) >> 3;
        b4 = fs_mask[(col + 1) & 7];


        /* prev row */
        p -= bmap->bpl;
        p[c2] |= b2;
        p[c3] |= b3;

        /* proper row */
        p += bmap->bpl;
        p[c1] |= b1;
        p[c2] |= b2;
        p[c3] |= b3;
        p[c4] |= b4;

        /* next row */
        p += bmap->bpl;
        p[c1] |= b1;
        p[c2] |= b2;
        p[c3] |= b3;
        p[c4] |= b4;

        /* final row */
        p += bmap->bpl;
        p[c2] |= b2;
        p[c3] |= b3;
    }
}

/****************************************************************
************ draw a line direct to bitmap **********************/
static FS_VOID draw_line_AA(_DS_ FS_FIXED x0, FS_FIXED y0, FS_FIXED x1, FS_FIXED y1)
{
    FS_FIXED dx, dy, x, y, xx, yy;


    /* degenerate line? set the pixel */
    dx = x1 - x0;
    dy = y1 - y0;
    if (dx == 0 && dy == 0)
    {
        set_pixel(_PS_ x0 >> 16, y0 >> 16);
        return;
    }

    {
        register FS_FIXED min, max, t;

        if (x1 > x0)
        {
            max = x1;
            min = x0;
        }
        else
        {
            max = x0;
            min = x1;
        }

        /* x == smallest half integer >= min */
        t = (min & 0xFFFF0000) + FIXED_ONEHALF;
        while (t < min)
            t += FIXED_ONE;
        x = t;

        /* xx == largest half integer <= max */
        t = (max & 0xFFFF0000) + FIXED_ONEHALF;
        while (t > max)
            t -= FIXED_ONE;
        xx = t;


        if (y1 > y0)
        {
            max = y1;
            min = y0;
        }
        else
        {
            max = y0;
            min = y1;
        }

        /* y == smallest half integer >= min */
        t = (min & 0xFFFF0000) + FIXED_ONEHALF;
        while (t < min)
            t += FIXED_ONE;
        y = t;

        /* yy == largest half integer <= max */
        t = (max & 0xFFFF0000) + FIXED_ONEHALF;
        while (t > max)
            t -= FIXED_ONE;
        yy = t;
    }

    /* any half pixels crossed ? */
    if (x > xx && y > yy)
    {

        return;    /* no */
    }

    /* a crossing in a single pixel .. set it and leave */
    if (FS_FLOOR(x0) == FS_FLOOR(x1) && FS_FLOOR(y0) == FS_FLOOR(y1))
    {
        set_pixel(_PS_ x0 >> 16, y0 >> 16);
        return;
    }

    /* ? horizontal */
    if (dy == 0)
    {
        register int n;

        y = y0 >> 16;
        x >>= 16;
        xx >>= 16;
        n = 1 + xx - x;
        while (n--)
        {
            set_pixel(_PS_ x, y);
            x++;
        }
        return;
    }

    /* ? vertical */
    if (dx == 0)
    {
        register int n;

        x = x0 >> 16;
        y >>= 16;
        yy >>= 16;
        n = 1 + yy - y;

        while (n--)
        {
            set_pixel(_PS_ x, y);
            y++;
        }
        return;
    }

    /* x major */
    {
        FS_FIXED grad, ex, ey, yey;
        register int n;

        grad = FixDiv(dy, dx);
        if (dx > 0)
        {
            ex = x0 - x;
            ey = FixMul(ex, grad);
            yey = y0 - ey;
        }
        else
        {
            ex = x1 - x;
            ey = FixMul(ex, grad);
            yey = y1 - ey;
        }
        x >>= 16;
        xx >>= 16;
        n = 1 + xx - x;

        while (n--)
        {
            set_pixel(_PS_ x, yey >> 16);
            x++;
            yey += grad;
        }
    }

    /* y-major */
    {
        FS_FIXED grad, xex, ex, ey;
        register int n;

        grad = FixDiv(dx, dy);
        if (dy > 0)
        {
            ey = y0 - y;
            ex = FixMul(ey, grad);
            xex = x0 - ex;
        }
        else
        {
            ey = y1 - y;
            ex = FixMul(ey, grad);
            xex = x1 - ex;
        }
        y >>= 16;
        yy >>= 16;
        n = 1 + yy - y;

        while (n--)
        {
            set_pixel(_PS_ xex >> 16, y);
            y++;
            xex += grad;
        }
    }
}

/****************************************************************/
/* midpoint algorithm for drawing a line */
static FS_VOID draw_line(_DS_ FS_FIXED x0, FS_FIXED y0, FS_FIXED x1, FS_FIXED y1)
{
    int ax, ay, bx, by, xinc, yinc, dx, dy;

    ax = FS_ROUND(x0);
    ay = FS_ROUND(y0);
    bx = FS_ROUND(x1);
    by = FS_ROUND(y1);
    dx = ABS(bx - ax);
    dy = ABS(by - ay);

    xinc = (ax > bx) ? -1 : 1;
    yinc = (ay > by) ? -1 : 1;

    if (dx > dy)
    {
        int px = dy << 1;
        int pxy = px - (dx << 1);
        int p = px - dx;

        for (; dx >= 0; dx--)
        {
            set_pixel(_PS_ ax, ay);
            if (p > 0)
            {
                ax += xinc;
                ay += yinc;
                p += pxy;
            }
            else
            {
                ax += xinc;
                p += px;
            }
        }
    }
    else
    {
        int py = dx << 1;
        int pxy = py - (dy << 1);
        int p = py - dy;

        for (; dy >= 0; dy--)
        {
            set_pixel(_PS_ ax, ay);
            if (p > 0)
            {
                ax += xinc;
                ay += yinc;
                p += pxy;
            }
            else
            {
                ay += yinc;
                p += py;
            }
        }
    }
}
/****************************************************************/
/* ceiling of the base 2 logarithm of x, ie: number of bits used */
static int log_2(FS_LONG x)
{
    int n = 0;

    if (x < 0)
        x = -x;

    while (x > (1 << n))
        n++;

    return n;
}

static FS_VOID bisect(_DS_ FS_FIXED fx1, FS_FIXED fy1, FS_FIXED fx2, FS_FIXED fy2, FS_FIXED fx3, FS_FIXED fy3)
{
    FS_FIXED lx, ly, rx, ry, mx, my;

    lx = (1 + fx1 + fx2) >> 1;
    rx = (1 + fx2 + fx3) >> 1;
    mx = (1 + lx + rx) >> 1;
    ly = (1 + fy1 + fy2) >> 1;
    ry = (1 + fy2 + fy3) >> 1;
    my = (1 + ly + ry) >> 1;
    draw_quad_sub(_PS_ fx1, fy1, lx, ly, mx, my);
    draw_quad_sub(_PS_ mx, my, rx, ry, fx3, fy3);
    return;
}
/****************************************************************/
/* incremental algorithm to draw a monotonic segment of a quadratic
*
* round the endpoints to integer, but the control points to half-integer
* since they only occur in the formulae multiplied by two ... might as
* well keep the additional accuracy since there's no extra cost.
*
* now we can translate the equations to (0,0), and derive the power
* series coefficients from the quadratic beziers as:
* x = 2*x2*t*(1-t) + x3*t*t
* y = 2*y2*t*(1-t) + y3*t*t
*
* x = 2*x2*t - 2*x2*t*t + x3*t*t
* y = 2*y2*t - 2*y2*t*t + y3*t*t
*
* x = (2*x2)*t + (x3 - 2*x2)*t*t
* y = (2*y2)*t + (y3 - 2*y2)*t*t
*
* x = a*t + b*t*t
* y = c*t + d*t*t
*
* and derive the coefficients of F(x,y) = Rxx + Sxy + Tyy + Ux + Vy
* take equations for x and y and eliminate t*t
* dx = da*t + db*t*t
* by = bc*t + bd*t*t
*
* subtract
* dx-by = (da-bc)*t
*
* t = (dx - by)/(da - bc);
* t = (dx-by)/K
*
* substitute this <t> into equation for x
* x = a*(dx-by)/K + b*(dx-by)(dx-by)/KK
*
* xKK = a*(dx-by)*k + b*(dx-by)*(dx-by)
*
* xKK = x*a*d*K - y*a*b*k + b*(dx*dx - 2*dx*by + by*by)
*
* xKK = x*adK - y*abK + xx*bdd - 2xy*bdb + yy*bbb
* 0 = x(adK-KK) - y*abK + xx*bdd - xy*bdb + yy*bbb
*
* But ... (adK-KK) == (ad-K)K == (ad-(da-bc))K == bcK
*
* 0 = x*bcK - y*abK + xx*bdd - 2xybdb +yy*bbb
*
* (assuming b != 0)
* 0 = x*cK - y*aK + x*dd - 2xy*bd + yy*bb
*       U      V       R       S       T
*
* Then do the usual partial differences. Since we're starting
* on the curve F=x=y=0;
*
* Fx(x,y) = F(x+dx,y)-F(x,y)
* Fx = R*(2*x*dx + dx*dx) + S*dx*y + U*dx
* Fx = R*(2*0*dx + 1) + S*dx*0 + U*dx
* Fx = R + U*dx
*
* Fxx(x,y) = Fx(x+dx,y)
* Fxx = 2*R*dx*dx
* Fxx = 2*R*1
* Fxx = R+R
*
*
* Fy(x,y) = F(x,y+dy) - F(x,y)
* Fy = T*(2*y*dy + dy*dy) + S*x*dy + V*dy
* Fy = T*(2*0*dy + 1) + S*0*dy + V*dy
* Fy = T + V*dy
*
* Fyy = Fy(x,y+dy) - Fy(x,y)
* Fyy = 2*T*dy*dy
* Fyy = 2*T*1
* Fyy = T+T
*
* Fxy(x,y) = Fx(x,y+dy) = Fy(x+dx,y) = Fyx(x,y)
* Fxy = S*dx*dy
*
*
* Now, so we can avoid worrying about what octant we're in
* and so we can get 8-connected curves, we look at all 3
* next candidate points (x+dx,y), (x,y+dy) and (x+dx,y+dy)
* and pick the point with minimal ABS(F).
*
* Also to make sure we don't jump sides of the parabola, test
* the sign of the partial derivatives for each point -- if they
* are different than the original point, reject that candidate.
*
*** I don't think the overflow logic is adequate, look some more
*** at the conditions for REALLY big characters
*/

static FS_VOID draw_quad_sub(_DS_ FS_FIXED fx1, FS_FIXED fy1, FS_FIXED fx2, FS_FIXED fy2, FS_FIXED fx3, FS_FIXED fy3)
{
    FS_SHORT x1, y1, x3, y3;
    FS_LONG temp, two_x2, two_y2;  /* might get larger than a short */
    FS_LONG a, b, c, d;
    FS_LONG R, S, T, U, V, cross;
    FS_SHORT dx, dy, loga, logc;
    FS_LONG F, Fx, Fy, Fxx, Fxy, Fyy;
    FS_LONG sx, sy, tx, ty;

    /* round the endpoints */
    x1 = FS_ROUND(fx1);            /* round start point to nearest integer  */
    y1 = FS_ROUND(fy1);
    temp = fx2 << 1;            /* round control point to nearest half-integer */
    two_x2 = FS_ROUND(temp);
    temp = fy2 << 1;
    two_y2 = FS_ROUND(temp);
    x3 = FS_ROUND(fx3);            /* round endpoint to nearest integer */
    y3 = FS_ROUND(fy3);

    /* OOPS ... this rounding can sometimes cause can cause non-monotonicity */
    if ((two_x2 > 2 * x1 && two_x2 > 2 * x3) || (two_x2 < 2 * x1 && two_x2 < 2 * x3))
        two_x2 = 2 * FS_ROUND(fx2);
    if ((two_y2 > 2 * y1 && two_y2 > 2 * y3) || (two_y2 < 2 * y1 && two_y2 < 2 * y3))
        two_y2 = 2 * FS_ROUND(fy2);

    /* the above correction might (again) cause a non-monotonic curve ... */
    if ((two_x2 > 2 * x1 && two_x2 > 2 * x3) || (two_x2 < 2 * x1 && two_x2 < 2 * x3))
    {
        bisect(_PS_ fx1, fy1, fx2, fy2, fx3, fy3);
        return;
    }
    if ((two_y2 > 2 * y1 && two_y2 > 2 * y3) || (two_y2 < 2 * y1 && two_y2 < 2 * y3))
    {
        bisect(_PS_ fx1, fy1, fx2, fy2, fx3, fy3);
        return;
    }
    /* check for a straight line */
    if ((2 * y3 - two_y2) * 2 * (x3 - x1) == (2 * x3 - two_x2) * 2 * (y3 - y1))
    {
        draw_line(_PS_ fx1, fy1, fx3, fy3);
        return;
    }

    /* power series coefficients  : x(t) = a*t + b*t*t, and y(t) = c*t + d*t*t   */
    a = two_x2 - (x1 << 1);
    b = (x3 - x1) - a;
    c = two_y2 - (y1 << 1);
    d = (y3 - y1) - c;

    /* check for overflow ... and maybe recurse */
    if (log_2(d) > 15 || log_2(b) > 15)
    {
        bisect(_PS_ fx1, fy1, fx2, fy2, fx3, fy3);
        return;
    }
    /* coefficients of F(x,y) = Rxx + Sxy + Tyy + Ux + Vy */
    R = d * d;
    S = - b * (d << 1);
    T = b * b;
    if (R == 0 && S == 0 && T == 0)
    {
        /* degenerate quadratic here ? */
        draw_line(_PS_ fx1, fy1, fx3, fy3);
        return;
    }

    cross = a * d - b * c;
    logc = (FS_SHORT)log_2(c);
    loga = (FS_SHORT)log_2(a);
    if ( (log_2(cross) + MAX(logc, loga)) > 30 )
    {
        bisect(_PS_ fx1, fy1, fx2, fy2, fx3, fy3);
        return;
    }
    U = c * cross;
    V = -a * cross;

    /* set the increments */
    dx = (x3 > x1) ? 1 : -1;
    dy = (y3 > y1) ? 1 : -1;

    /* initial F() and other partial differences */
    F = 0;
    Fx = R + U * dx;
    Fxx = R + R;
    Fxy = S * dx * dy;
    Fy = T + V * dy;
    Fyy = T + T;

    /* keep track of sign of first derivatives too ... they should
     * not change if we're staying on the same 'leg' of the parabola */
    sx = Fx - R;
    sy = Fy - T;

    /* mark some pixels */
    set_pixel(_PS_ x1, y1);
    while (x1 != x3 || y1 != y3)
    {
        FS_LONG f1, f2, f3;

        /* horizontal move: new derivatives would be... */
        tx = Fx + Fxx - R;
        ty = Fy + Fxy - T;
        if ((sx < 0 && tx > 0) || (sx > 0 && tx < 0) || (sy < 0 && ty > 0) || (sy > 0 && ty < 0))
            f1 = 0x7FFFFFFF;
        else
            f1 = F + Fx;

        /* vertical move: new derivatives would be... */
        tx = Fx + Fxy - R;
        ty = Fy + Fyy - T;
        if ((sx < 0 && tx > 0) || (sx > 0 && tx < 0) || (sy < 0 && ty > 0) || (sy > 0 && ty < 0))
            f2 = 0x7FFFFFFF;
        else
            f2 = F + Fy;

        /* diagonal move: new derivatives would be... */
        tx = Fx + Fxx + Fxy - R;
        ty = Fy + Fyy + Fxy - T;
        if ((sx < 0 && tx > 0) || (sx > 0 && tx < 0) || (sy < 0 && ty > 0) || (sy > 0 && ty < 0))
            f3 = 0x7FFFFFFF;
        else
            f3 = F + Fx + Fy + Fxy;


        /* really interested in the absolute values */
        f1 = ABS(f1);
        f2 = ABS(f2);
        f3 = ABS(f3);

        /* prefer the diagonal move all other things being equal */
        if (f3 <= f1 && f3 <= f2)
        {
            x1 += dx;
            F += Fx;
            Fx += Fxx;
            Fy += Fxy;
            y1 += dy;
            F += Fy;
            Fx += Fxy;
            Fy += Fyy;
        }
        else if (f1 <= f2 && f1 <= f3) /* then x */
        {
            x1 += dx;
            F += Fx;
            Fx += Fxx;
            Fy += Fxy;
        }
        else /* then y */
        {
            y1 += dy;
            F += Fy;
            Fx += Fxy;
            Fy += Fyy;
        }
        set_pixel(_PS_ x1, y1);
    }
    return;
}
/****************************************************************/
/* split a quadratic up into monotonic pieces -- they usually are
 * monotonic; so the speed of the tests is important, although the
 * speed of the actual splitting code is less interesting */
static FS_VOID draw_quad(_DS_ FS_FIXED x1, FS_FIXED y1, FS_FIXED x2, FS_FIXED y2, FS_FIXED x3, FS_FIXED y3)
{
    /* ? y switches direction internally -- need to split */
    if ((y2 > y1 && y2 > y3) || (y2 < y1 && y2 < y3))
    {
        FS_FIXED num, denom, Lx, Rx, xs, ys;

        num = y2 - y1;
        denom = (y2 - y1) + (y2 - y3);
        ys = y1 + LongMulDiv(y2 - y1, num, denom);
        Lx = x1 + LongMulDiv(x2 - x1, num, denom);
        Rx = x2 + LongMulDiv(x3 - x2, num, denom);
        xs = Lx + LongMulDiv(Rx - Lx, num, denom);
        draw_quad(_PS_ x1, y1, Lx, ys, xs, ys);
        draw_quad(_PS_ xs, ys, Rx, ys, x3, y3);
    }

    /* ? x switches direction internally -- need to split */
    else if ((x2 > x1 && x2 > x3) || (x2 < x1 && x2 < x3))
    {
        FS_FIXED num, denom, Ly, Ry, ys, xs;

        num = x2 - x1;
        denom = (x2 - x1) + (x2 - x3);
        xs = x1 + LongMulDiv(x2 - x1, num, denom);
        Ly = y1 + LongMulDiv(y2 - y1, num, denom);
        Ry = y2 + LongMulDiv(y3 - y2, num, denom);
        ys = Ly + LongMulDiv(Ry - Ly, num, denom);

        draw_quad(_PS_ x1, y1, xs, Ly, xs, ys);
        draw_quad(_PS_ xs, ys, xs, Ry, x3, y3);
    }
    else
    {
        /* it's finally monotonic */
        draw_quad_sub(_PS_ x1, y1, x2, y2, x3, y3);
    }
}

/****************************************************************
****** draw an 8-connected quadratic direct to bitmap **********/
static FS_VOID draw_quad_AA(_DS_ FS_FIXED x1, FS_FIXED y1, FS_FIXED x2, FS_FIXED y2, FS_FIXED x3, FS_FIXED y3)
{
    FS_FIXED lx, ly, rx, ry, mx, my, cx, cy;
    FS_FIXED alx, aly, brx, bry;
    FS_FIXED x, y;
    FS_FIXED err, limit, DistanceNorm(FS_FIXED, FS_FIXED);

    /* any half pixels crossed ? */
    lx = MIN(x1, MIN(x2, x3));
    rx = MAX(x1, MAX(x2, x3));
    ly = MIN(y1, MIN(y2, y3));
    ry = MAX(y1, MAX(y2, y3));

    /* following check does not include half integers. That is the
       correct behavior for raster centers ... but not line drawing
       if ((rx < ABOVE(lx) || lx > BELOW(rx)) && (ry < ABOVE(ly) || ly > BELOW(ry))) */

    /* alx == smallest half integer >= lx */
    alx = (lx & 0xFFFF0000) + FIXED_ONEHALF;
    while (alx < lx)
        alx += FIXED_ONE;

    /* aly == smallest half integer >= ly */
    aly = (ly & 0xFFFF0000) + FIXED_ONEHALF;
    while (aly < ly)
        aly += FIXED_ONE;

    /* bhx == largest half integer <= rx */
    brx = (rx & 0xFFFF0000) + FIXED_ONEHALF;
    while (brx > rx)
        brx -= FIXED_ONE;

    /* bry == largest half integer <= ry */
    bry = (ry & 0xFFFF0000) + FIXED_ONEHALF;
    while (bry > ry)
        bry -= FIXED_ONE;

    if ((rx < alx || lx > brx) && (ry < aly || ly > bry))
    {
        return;    /* no */
    }

    /* all coordinates in the same pixel? */
    x = FS_FLOOR(x1);
    y = FS_FLOOR(y1);
    if (x == FS_FLOOR(x2) && x == FS_FLOOR(x3) && y == FS_FLOOR(y2) && y == FS_FLOOR(y3))
    {
        /* well ... since we know we cross a half pixel, it must be here */
        set_pixel(_PS_ x, y);
        return;
    }

    /* subdivide the quadratic */
    lx = (x1 + x2) >> 1;
    ly = (y1 + y2) >> 1;
    rx = (x2 + x3) >> 1;
    ry = (y2 + y3) >> 1;
    mx = (lx + rx) >> 1;
    my = (ly + ry) >> 1;    /* parametric midpoint */
    cx = (x1 + x3) >> 1;
    cy = (y1 + y3) >> 1;    /* chord midpoint */

    /* same constructions as in FS_quad */
    limit = (FS_FIXED)STATE.lpm << 8;
    err = DistanceNorm(mx - cx, my - cy);
    if (err > limit)
    {
        draw_quad_AA(_PS_ x1, y1, lx, ly, mx, my);
        draw_quad_AA(_PS_ mx, my, rx, ry, x3, y3);
    }
    else
    {
        draw_line_AA(_PS_ x1, y1, mx, my);
        draw_line_AA(_PS_ mx, my, x3, y3);
    }
}


/****************************************************************/
/** draw the stik directly into the bitmap ... what a concept ***/
/* outl passed in will be deleted by caller chain (make_bitmap) */
FS_VOID draw_stik(_DS_ FS_OUTLINE *outl)
{
    FS_SHORT i, num = outl->num;
    FS_BYTE *type = (FS_BYTE *)(outl->type);
    FS_FIXED x0 = 0, y0 = 0, x1, y1, x2, y2;
    FS_FIXED *x = (FS_FIXED *)(outl->x);
    FS_FIXED *y = (FS_FIXED *)(outl->y);
    FS_SHORT sw;
    FS_FIXED delta = 0;
    FS_BOOLEAN need_to_free_outl = 0;


    if ( !(STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON))
    {
        FS_OUTLINE *ah_stik;
        /* outl passed in will be deleted by caller chain (make_bitmap) */
        /* we need to free ah_stik when we are done with it */
        ah_stik = copy_outline(_PS_ outl, 0);
        if (ah_stik)
        {
            /* copy is successful */
            need_to_free_outl = 1;
            outl = ah_stik;
            type = (FS_BYTE *)(outl->type);
            x = (FS_FIXED *)(outl->x);
            y = (FS_FIXED *)(outl->y);
        }
        autohint_stik(_PS_ outl, 0);
        if (STATE.error)
        {
            if (need_to_free_outl) FSS_free_char(_PS_ outl);
            return;
        }
    }

    /* grid alignment needs to be half grid for FS_direct drawing */
    /* but the autohinter doesn't know that ... */
    /*    sw = FS_ROUND(STATE.lpm * STATE.stroke_pct);
     *    sw = MAX(1,sw);
     *    if (sw==2 || sw==4)
     *        delta = FIXED_ONEHALF;
     *    else
     *        delta = 0;
     */


    sw = FS_ROUND(STATE.lpm * STATE.stroke_pct);
    sw = MAX(1, sw);
    if (sw == 2 || sw == 4)
        delta = FIXED_ONEHALF;


    for (i = 0; i < num; i++)
    {
        switch (type[i] & 0x7F)
        {
        case FS_MOVETO:
            x0 = delta + *x++;
            y0 = delta + *y++;
            break;
        case FS_LINETO:
            x1 = delta + *x++;
            y1 = delta + *y++;
            if (STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON)
                draw_line_AA(_PS_ x0, y0, x1, y1);
            else
                draw_line(_PS_ x0, y0, x1, y1);

            x0 = x1;
            y0 = y1;
            break;
        case FS_QUADTO:
            x1 = delta + *x++;
            y1 = delta + *y++;
            x2 = delta + *x++;
            y2 = delta + *y++;
            if (STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON)
                draw_quad_AA(_PS_ x0, y0, x1, y1, x2, y2);
            else
                draw_quad(_PS_ x0, y0, x1, y1, x2, y2);
            x0 = x2;
            y0 = y2;
            break;
        default:
            break;
        }
    }
    if (need_to_free_outl) FSS_free_char(_PS_ outl);
}
/****************************************************************/
#endif /* FS_STIK */
#endif
